<?php

namespace App\Http\Controllers;

use App\Models\Bank;
use App\Models\City;
use App\Models\District;
use JWTAuth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Tymon\JWTAuth\Exceptions\JWTException;
use Illuminate\Support\Facades\Validator;

class APIController extends Controller
{
    public function authenticate(Request $request)
    {
        $credentials = $request->only('email', 'password');

        //valid credential
        $validator = Validator::make($credentials, [
            'email' => 'required|email',
            'password' => 'required|string|min:6|max:50'
        ]);

        //Send failed response if request is not valid
        if ($validator->fails()) {
            return response()->json(['error' => $validator->messages()], 200);
        }

        //Request is validated
        //Crean token
        try {
            if (! $token = JWTAuth::attempt($credentials)) {
                return response()->json([
                	'success' => false,
                	'message' => 'Login credentials are invalid.',
                ], 400);
            }
        } catch (JWTException $e) {
    	return $credentials;
            return response()->json([
                	'success' => false,
                	'message' => 'Could not create token.',
                ], 500);
        }
 	
 		//Token created, return with success response and jwt token
        if(JWTAuth::user()->is_admin == 3)
        {
            return response()->json([
                'success' => true,
                'token' => $token,
            ]);
        }
        else
        {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized.',
            ], 400);
        }
    }

    public function getPostInfo(Request $request){
        $ziraatKatilim = Bank::where('code', 'ziraatkatilim')->first();
        
        $jsonData = [];
        $posts = $ziraatKatilim->posts;
        if(!$posts->isEmpty())
        {
        	foreach($posts as $post) {
	            switch($post->postable->property_type)
	            {
	                case 0:
	                    $subCategory = config('enums.house_type')[$post->postable->house_type];
	                    break;
	                case 1:
	                    $subCategory = config('enums.workplace_type')[$post->postable->workplace_type];
	                    break;
	                case 2:
	                    $subCategory = config('enums.land_type')[$post->postable->land_type];
	            }
	            $jsonData[] = [
	                'no' => $post->code,
	                'name' => $post->title,
	                'category' => config('enums.property_type')[$post->postable->property_type],
	                'subCategory' => $subCategory,
	                'city' => $post->postable->country,
	                'cityId' => City::where('name', $post->postable->country)->first()->id,
	                'county' => $post->postable->district,
	                'countytId' => District::where('name', $post->postable->district)->first()->id,
	                'district' => $post->postable->neighborhood,
	                'districtId' => $post->postable->neighborhood_id,
	                'minArtis' => $post->offer_step,
	                'baslangicBedeli' => $post->start_price,
	                'detailUrl' => "http://ziraatkatilim.propturk.com/ihaleler/".$post->postable->slug,
	                'pafta' => null,
	                'ada' => $post->postable->city_block_no,
	                'parsel' => $post->postable->parcel_no,
	                'creditAvailable' => $post->postable->is_creditable ? "Uygun" : "Uygun değil",
	                'constructionYear' => now()->year - $post->postable->building_age,
	                'area' => $post->postable->gross_m2,
	                'usageStatus' => isset($post->postable->using_status) ? config('enums.using_status')[$post->postable->using_status] : null,
	                'heatingType' => config('enums.heating_type')[isset($post->postable->heating_type) ? $post->postable->heating_type : 0],
	                'deedStatus' => null,
	                'floor' => isset($post->postable->floor) ? config('enums.floor')[$post->postable->floor] : null,
	                'buildingType' => null,
	                'buildingState' => null,
	                'numberOfRooms' => $post->postable->room_count,
	                'numberOfLeavingRooms' => null,
	                'imageUrls' => $this->getPhotoUrlArray($post)
	            ];
        	}
        	// dd($jsonData);
        	return response()->json($jsonData, 200, ['Content-type'=>'application/json;charset=utf-8'], JSON_UNESCAPED_UNICODE);
        }
        else
        {
        	return response()->json([
                	'message' => 'Bankanıza ait gayrimenkul bulunmamaktadır.',
                ], 200, ['Content-type'=>'application/json;charset=utf-8'], JSON_UNESCAPED_UNICODE);
        }
    }

    protected function getPhotoUrlArray($post)
    {
        $postImageUrl = storage_path('app/public/'.$post->images_folder);
        // dd(asset('assets/img/turkiye-finans/sand-watch.png'));
        $imagesPaths = array_diff(scandir($postImageUrl), array('.', '..'));
        $imageUrls = [];
        foreach($imagesPaths as $imagesPath)
        {
            // dd($imagesPath);
            // dd(Storage::url($imagesPath));
            // $imageUrls[] = storage_path('app/public/'.$post->images_folder)."/".$imagesPath;
            $imageUrls[] = asset(Storage::url($post->images_folder.'/'.$imagesPath));
        }
        return $imageUrls;
    }
}
