<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Mail\UserRegisterMail;
use App\Models\NotificationLog;
use App\Providers\RouteServiceProvider;
use App\Models\User;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Cviebrock\EloquentSluggable\Services\SlugService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {

        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'surname' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'phone_number' => ['required', 'unique:users'],
            'company_name' => ['nullable', 'string'],
            'company_tax_office' => ['nullable', 'string'],
            'company_tax_id' => ['nullable', 'string'],
            'g-recaptcha-response' => 'required|recaptcha'
        ]);
       
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $phoneNumber =  substr(preg_replace('/[^0-9]/', '', $data['phone_number']), 1);
        
        if(!isset($data['sms_permission'])) {
            $data['sms_permission'] = 0;
        }
        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
            'surname' => $data['surname'],
            'phone_number' => $phoneNumber,
            'slug'  => SlugService::createSlug(User::class, 'slug', $data['name'].'-'.$data['surname'].'-'.Str::random(5)),
            'sms_permission' => $data['sms_permission'],
            'company_name' => $data['company_name'],
            'company_tax_office' => $data['company_tax_office'],
            'company_tax_id' => $data['company_tax_id'],
        ]);

        $notificationLogType = 'mail';
        $notificationLogName = 'Mail - Yeni Kullanıcı kayıt bildirimi'; //UserRegisterMail
        NotificationLog::createNotificationLog(NULL, $user, $notificationLogType, $notificationLogName, NULL);

        Mail::to('bilgi@markest.com.tr')->queue(
            new UserRegisterMail($user)
        );

        Log::channel('Yeni-Kullanıcı-Kayıt')->info('KULLANICI KAYIT~'.$user->email." - ".$user->name." ".$user->surname." "."kayıt oldu.");

        
        return $user; 
    }
}